/*
* ReaderEntryDislpay.cpp
*
*  Created on: 03.11.2009
*      Author: stefan.detter
*/

#include "ReaderEntry.h"

#include <QFont>
#include <QPixmap>
#include <QBrush>

#include "../../application/Reader.h"

#include "../../permission/Permissions.h"


TreeItem::TYPE ReaderEntry::itemType() const
{
	return READER;
}

QVariant ReaderEntry::data(int column, int role) const
{
	switch(role)
	{
		case Qt::TextAlignmentRole:
		{
			if((TreeItem::COLUMN_INDEX)column == COL_ID)
				return int(Qt::AlignLeft | Qt::AlignVCenter);
			else
				return int(Qt::AlignHCenter | Qt::AlignVCenter);

			break;
		}

		case Qt::DisplayRole:
		{
			switch((TreeItem::COLUMN_INDEX)column)
			{
			case COL_ID:
				return m_reader->name();
			case COL_TagType:
				if(m_isOffline)
					return "Offline";
				return m_reader->tagTypeString();
			case COL_ReadCount:
				return QString::number(m_readTagCount);
			case COL_ReadRate:
				return QString("%1").arg(m_readRate, 1, 'g', 5);
			case COL_Action:
				if(m_isOffline)
					return "Offline";
				return m_reader->currentActionString();
			case COL_State:
				if(m_isOffline)
					return "Offline";
				return m_reader->currentStateString();
			}
			break;
		}

		case Qt::FontRole:
		{
			QFont f = m_treeBoxFont;
			f.setBold(true);
			switch((TreeItem::COLUMN_INDEX)column)
			{
			case COL_ID:
#ifdef Q_OS_WIN
				f.setPointSize(11);
#else
				f.setPointSize(13);
#endif
				break;
				//case COL_Action:
			case COL_State:
#ifdef Q_OS_WIN
				f.setPointSize(10);
#else
				f.setPointSize(12);
#endif
				break;
			default:
#ifdef Q_OS_WIN
#else
				f.setPointSize(11);
#endif
				break;

			}

			return f;
			break;
		}

		case Qt::DecorationRole:
		{
			switch((TreeItem::COLUMN_INDEX)column)
			{
			case COL_ID:
				return QPixmap(":/icons/rfe_icon");
			case COL_State:
				switch(m_reader->currentState())
				{
				case QrfeGlobal::STATE_ONLINE:
					return *m_onlineIcon;
				default:
					break;
				}
				break;
			default:
				break;
			}
			break;
		}

		case Qt::BackgroundRole:
		{
			return headingBackgroundColor();
			break;
		}

		case Qt::ForegroundRole:
		{
			QColor c(Qt::white);
			switch((TreeItem::COLUMN_INDEX)column)
			{
			case COL_Action:
				switch(m_reader->currentAction())
				{
				case QrfeGlobal::ACTION_IDLE:
					c = QColor(0xFF, 0xE6, 0x82);
					break;
				case QrfeGlobal::ACTION_SCANNING:
					c = QColor(Qt::cyan);
					break;
				case QrfeGlobal::ACTION_OFFLINE:
				case QrfeGlobal::ACTION_WAITING:
				default:
					c = QColor(Qt::red);
				}
				break;
			case COL_State:
				switch(m_reader->currentState())
				{
				case QrfeGlobal::STATE_ONLINE:
					c = QColor(Qt::green);
					break;
				case QrfeGlobal::STATE_NOTRESPONDING:
				case QrfeGlobal::STATE_OFFLINE:
				default:
					c = QColor(Qt::red);
					break;
				}
				break;
			default:
				break;
			}
			return QBrush(c);
			break;
		}
	}

	return QVariant();
}


int ReaderEntry::informationRowCount () const
{
	if(m_reader->currentSystem() == QrfeGlobal::FIRMWARE)
		return m_informationValues.size();
	else
		return READER_ROW_APPLICATION;
}

QVariant ReaderEntry::information (int row, int column, int role) const
{
	switch(role)
	{
		case Qt::DecorationRole:
		{
			if(column == 0)
			{
				switch(row)
				{
				case READER_ROW_HEADER:
					return QPixmap(":/icons/rfe_icon");
				}
			}
			break;
		}

		case Qt::FontRole:
		{
			if(column == 1)
			{
				QFont f = m_informationBoxFont;
				f.setBold(false);
				switch(row)
				{
				case READER_ROW_System:
					if(m_reader->currentSystem() == QrfeGlobal::BOOTLOADER)
						f.setBold(true);
					break;

				case READER_ROW_State:
					if( m_reader->currentState() == QrfeGlobal::STATE_OFFLINE ||
						m_reader->currentState() == QrfeGlobal::STATE_NOTRESPONDING)
						f.setBold(true);
					break;
				default:
					break;
				}
#ifdef Q_OS_WIN
#else
				f.setPointSize(11);
#endif
				return f;
			}
			break;
		}

		case Qt::ForegroundRole:
		{
			if(column == 1)
			{
				Qt::GlobalColor color = Qt::black;
				switch(row)
				{
				case READER_ROW_Action:
					switch(m_reader->currentAction())
					{
					case QrfeGlobal::ACTION_IDLE:
						color = Qt::darkYellow;
						break;
					case QrfeGlobal::ACTION_SCANNING:
						color = Qt::darkBlue;
						break;
					case QrfeGlobal::ACTION_OFFLINE:
					case QrfeGlobal::ACTION_WAITING:
					default:
						color = Qt::darkRed;
						break;
					}
					break;
				case READER_ROW_State:
					switch(m_reader->currentState())
					{
					case QrfeGlobal::STATE_ONLINE:
						color = Qt::darkGreen;
						break;
					case QrfeGlobal::STATE_NOTRESPONDING:
					case QrfeGlobal::STATE_OFFLINE:
					default:
						color = Qt::darkRed;
						break;
					}
					break;
				}
				return QBrush(color);
			}
			break;
		}

		case Qt::ToolTipRole:
		{
			if(column == 1)
			{
				switch(row)
				{
				case READER_ROW_StatusRegister:
					if(Permissions::d->readerAdvancedSettings())
					{
						QStringList descs = m_reader->statusRegisterBitDescriptions();
						return descs.join("\n");
					}
					break;
				case READER_ROW_SWRev:
                    if(Permissions::d->readerAdvancedSettings() && m_reader->currentSystem() == QrfeGlobal::FIRMWARE )
					{
						return m_reader->advancedSoftwareRevisionString();
					}
					break;
                case READER_ROW_BLRev:
                    if(Permissions::d->readerAdvancedSettings() && m_reader->currentSystem() == QrfeGlobal::BOOTLOADER )
                    {
                        return m_reader->advancedSoftwareRevisionString();
                    }
                    break;
                }
			}
			break;
		}
	}

	return TreeItem::information(row, column, role);
}

void ReaderEntry::prepareInformation ( )
{
	int size = 27;

	m_informationHeader.clear();
	m_informationHeader << READER_ROW_HEADER;

	m_informationSubHeader.clear();
	m_informationSubHeader << READER_ROW_COMMON << READER_ROW_APPLICATION << READER_ROW_CURRENTREAD << READER_ROW_DYNAMIC_INFO;

	m_informationKeys.resize(size);
	m_informationKeys[READER_ROW_HEADER]			= "Reader";
	m_informationKeys[READER_ROW_COMMON]			= "Common";
	m_informationKeys[READER_ROW_SerialNumber]		= "Reader Serial";
	m_informationKeys[READER_ROW_SerialNumberDec]	= "Reader Serial Dec";
	m_informationKeys[READER_ROW_ReaderModule]		= "Reader Module";
	m_informationKeys[READER_ROW_AntennaCount]		= "Antenna Count";
	m_informationKeys[READER_ROW_DeviceName]		= "Interface";
	m_informationKeys[READER_ROW_ReaderType]		= "Reader Type";
	m_informationKeys[READER_ROW_HWRev]				= "Hardware Revision";
	m_informationKeys[READER_ROW_SWRev]				= "Software Revision";
	m_informationKeys[READER_ROW_BLRev]				= "Bootloader Revision";
	m_informationKeys[READER_ROW_TagType]			= "Current Tag Type";
	m_informationKeys[READER_ROW_System]			= "System";
	m_informationKeys[READER_ROW_Action]			= "Action";
	m_informationKeys[READER_ROW_State]				= "State";
	m_informationKeys[READER_ROW_StatusRegister]	= "Status Register";
	m_informationKeys[READER_ROW_GPIOPins]			= "GPIO Pins";
	m_informationKeys[READER_ROW_AttachedTime]		= "Attached at";
	m_informationKeys[READER_ROW_APPLICATION]		= "Application";
	m_informationKeys[READER_ROW_Name]				= "Reader Name";
	m_informationKeys[READER_ROW_Location]			= "Location";
	m_informationKeys[READER_ROW_CURRENTREAD]		= "Current Read";
	m_informationKeys[READER_ROW_DifferentTags]		= "Different Tags";
	m_informationKeys[READER_ROW_ReadCount]			= "Read Count";
	m_informationKeys[READER_ROW_ReadRate]			= "Read Rate";
	m_informationKeys[READER_ROW_MaxReadRate]		= "Max. Read Rate";
	m_informationKeys[READER_ROW_DYNAMIC_INFO]		= "Dynamic Informations";

	m_informationValues.resize(size);
	m_informationValues[READER_ROW_SerialNumber] 	= m_reader->serialNumber();
	m_informationValues[READER_ROW_SerialNumberDec]	= QString::number(m_reader->serialNumberExact());
	m_informationValues[READER_ROW_ReaderType] 		= m_reader->readerTypeExactString();
	m_informationValues[READER_ROW_ReaderModule] 	= m_reader->readerModuleString();
	m_informationValues[READER_ROW_AntennaCount]	= QString::number(m_reader->antennaCount());
	m_informationValues[READER_ROW_DeviceName] 		= m_reader->deviceName();
	m_informationValues[READER_ROW_HWRev] 			= m_reader->hardwareRevisionString();
	m_informationValues[READER_ROW_SWRev] 			= m_reader->softwareRevisionString();
	m_informationValues[READER_ROW_BLRev] 			= m_reader->bootloaderRevisionString();
	m_informationValues[READER_ROW_TagType] 		= m_reader->tagTypeString();
	m_informationValues[READER_ROW_System] 			= m_reader->currentSystemString();
	m_informationValues[READER_ROW_Action] 			= m_reader->currentActionString();
	m_informationValues[READER_ROW_State] 			= m_reader->currentStateString();
	m_informationValues[READER_ROW_StatusRegister] 	= QString("0x%1").arg(m_reader->statusRegister(), 16, 16, QChar('0'));
	m_informationValues[READER_ROW_GPIOPins] 		= QString("0x%1").arg(m_reader->gpioValues(), 8, 16, QChar('0'));
	m_informationValues[READER_ROW_AttachedTime] 	= m_attachedTime.toString(Qt::ISODate);
	m_informationValues[READER_ROW_Name]		 	= m_reader->readerName();
	m_informationValues[READER_ROW_Location]	 	= m_reader->readerLocation();
	updateInformation ( );
}

void ReaderEntry::updateInformation ( )
{
	if(m_informationValues.size() == 0)
		return;

	QVector<QString> temp(m_informationValues);

	m_informationValues[READER_ROW_DifferentTags] = QString::number(tagCount());
	m_informationValues[READER_ROW_ReadCount] = 	QString::number(m_readTagCount);
	m_informationValues[READER_ROW_ReadRate] = 		QString("%1").arg(m_readRate, 1, 'g', 5);
	m_informationValues[READER_ROW_MaxReadRate] = 	QString("%1").arg(m_maxReadRate, 1, 'g', 5);

	if(temp != m_informationValues)
		emit informationChanged(READER_ROW_DifferentTags, READER_ROW_MaxReadRate);
}

QBrush ReaderEntry::headingBackgroundColor() const
{
	QLinearGradient lg(0, 0, 0, 50 );
	switch(m_reader->currentState())
	{
	case QrfeGlobal::STATE_ONLINE:
		if(m_reader->currentSystem() == QrfeGlobal::BOOTLOADER)
			lg.setColorAt(0, Qt::darkRed);
		else
		{
			if(!m_scanRunning)
				lg.setColorAt(0, QColor(0x0C, 0x08, 0x5C));
			else
				lg.setColorAt(0, QColor(33, 158, 255));
		}
		break;
	default:
		lg.setColorAt(0, Qt::black);
		break;
	}
	lg.setColorAt(1, Qt::white);
	return QBrush(lg);
}



